package de.dwds.ddc.client;

import java.io.*;
import java.net.*;

/** low-level class for DDC server/client protocol socket communications; see java.net.Socket */
public class DDCSocket {
	private String host;
	private int    port;
	private int    timeout_ms;
	private Socket sock;
	
	//-----------------------------------------------------------------
	// constructors etc
	/** default constructor; use connect() to connect the socket */
	public DDCSocket() {
		this.host="";
		this.port=0;
		this.timeout_ms=0;
		this.sock=null;
	}
	
	/** constructor given host and port */
	public DDCSocket(String host, int port)
	{
		this.host=host;
		this.port=port;
		this.timeout_ms=0;
		this.sock=null;
	}

	/** constructor given host, port and timeout */
	public DDCSocket(String host, int port, int timeout_ms)
	{
		this.host=host;
		this.port=port;
		this.timeout_ms=timeout_ms;
		this.sock=null;
	}

	
	//-----------------------------------------------------------------
	// accessors
	/** @return the currently selected remote host */
	public String getHost() {
		return host;
	}

	/** @return the currently selected remote port */
	public int getPort() {
		return port;
	}
	
	/** @return the currently selected timeout, in milliseconds (0: none) */
	public int getTimeout() {
		return timeout_ms;
	}

	/** @return the current socket, if any*/
	public Socket getSocket() {
		return sock;
	}

	//-----------------------------------------------------------------
	// open/close
	/** return true iff connection is currently open */
	public boolean isConnected() {
		return sock != null && sock.isConnected();
	}
	
	/** close current connection, if any 
	 * @throws IOException */
	public void close() throws IOException {
		if (isConnected()) { sock.close(); }
		sock = null;
	}

	/** (re-)open connection to a (new) server
	 * @throws IOException 
	 * @throws UnknownHostException */
	public void connect(String host, int port)
			throws UnknownHostException, IOException
	{
		this.host = host;
		this.port = port;
		connect();
	}

	/** (re-)open connection to the currently selected server */
	public void connect() 
			throws UnknownHostException, IOException
	{
		close();
		sock = new Socket(host,port);
		sock.setSoTimeout(timeout_ms);
		//sock.setKeepAlive(true);
	}
	
	/** set timeout on underlying socket; wraps use getSocket().setSoTimeout(timeout_ms)
	 * @param timeout_ms read timeout in milliseconds
	 */
	public void setTimeout(int timeout_ms) throws IOException
	{
		this.timeout_ms = timeout_ms;
		if (isConnected()) {
			sock.setSoTimeout(timeout_ms);
		}
	}

	//-----------------------------------------------------------------
	// low-level socket communications

	/** send message bytes over the socket (socket must be connected) 
	 * @throws IOException */
	public void send(byte[] msg)
			throws IOException
	{
		if (!isConnected()) {
			throw new IOException("cannot send() to unconnected socket");
		}
		OutputStream os = sock.getOutputStream();
		byte[] lmsg = DDCUtils.sizeToBytes(msg.length);
		os.write(lmsg);
		os.write(msg);
		os.flush();
	}
	
	/** read a message from the socket (socket must be connected)
	 * @throws IOException */
	public byte[] receive() throws IOException
	{
		if (!isConnected()) {
			throw new IOException("cannot receive() from unconnected socket");
		}

		//-- read size
		//InputStream is = sock.getInputStream(); 
		DataInputStream is = new DataInputStream( sock.getInputStream() );
		byte[] sizebuf = {0,0,0,0};
		is.readFully(sizebuf);
		int msglen = DDCUtils.bytesToSize(sizebuf);
		
		//-- read message body
		byte[] msgbuf = new byte[msglen];
		is.readFully(msgbuf);
		return msgbuf;
	}
		
	/** send a request and wait for a response; wraps connect(), send(), receive(), close() 
	 * @throws IOException */
	public byte[] request(byte[] msg) throws IOException
	{
		if (!isConnected()) { connect(); }
		this.send(msg);
		byte[] buf = this.receive();
		if (isConnected()) { close(); }
		return buf;
	}

	//-----------------------------------------------------------------
	// socket communications: string wrappers

	/** send a string message over the socket; wraps this.send(msg.getBytes()) */
	public void send(String msg) throws IOException
	{ this.send(msg.getBytes()); }

	/** read a String message from the socket; wraps new String(this.receive())
	 * @throws IOException */
	public String receiveString() throws IOException
	{ return new String( this.receive() ); }

	/** send a string request and wait for a response; wraps new String(this.request(msg.getBytes()))
	 * @throws IOException */
	public String request(String msg) throws IOException
	{ return new String( this.request(msg.getBytes()) );  }

}
